import binascii
import io
import os
import re
import sys
import time
import zlib
import struct
import tkinter as tk
from tkinter import ttk, filedialog, messagebox
from urllib import request, parse, error
from html.parser import HTMLParser
from dataclasses import dataclass
from typing import Dict, List, Tuple, Optional

DEFAULT_SERVER = "http://ikw.ddns.net"
FOLDER_MAP_TXT = "FolderToTrackName.txt"
VALID_MODES = ["150", "150F", "200", "200F"]
RKG_HEADER_SIZE = 0x88
RKG_MAGIC = b"RKGD"
YAZ1_MAGIC = b"YAZ1"
INPUT_MAX_UNCOMPRESSED = 0x2774

SAFE_FILENAME_RE = re.compile(r"^[A-Za-z0-9._-]+$")

def http_get(url: str, timeout: float = 15.0) -> bytes:
    req = request.Request(url, headers={"User-Agent": "IKWGhostManager/1.0"})
    with request.urlopen(req, timeout=timeout) as resp:
        return resp.read()

def http_list(url: str, timeout: float = 15.0) -> List[str]:
    try:
        data = http_get(parse.urljoin(url, "index.json"), timeout=timeout)
        import json
        arr = json.loads(data.decode("utf-8", "replace"))
        out = []
        for entry in arr:
            name = entry.get("name")
            if name:
                out.append(name)
        return out
    except Exception:
        pass
    class _DirParser(HTMLParser):
        def __init__(self):
            super().__init__()
            self.links: List[str] = []
        def handle_starttag(self, tag, attrs):
            if tag.lower() == "a":
                href = None
                for k, v in attrs:
                    if k.lower() == "href":
                        href = v
                        break
                if href and href not in ("/", "..", "."):
                    self.links.append(href)
    try:
        html = http_get(url, timeout=timeout).decode("utf-8", "replace")
        p = _DirParser()
        p.feed(html)
        cleaned = []
        for h in p.links:
            h = h.split("#")[0].split("?")[0]
            name = h.rstrip("/").split("/")[-1]
            if name:
                cleaned.append(name)
        return sorted(set(cleaned))
    except Exception:
        return []

@dataclass
class RkgInfo:
    valid: bool
    reason: str
    compressed: Optional[bool] = None
    finish_time_ms: Optional[int] = None
    track_id: Optional[int] = None
    vehicle_id: Optional[int] = None
    character_id: Optional[int] = None
    year: Optional[int] = None
    month: Optional[int] = None
    day: Optional[int] = None
    controller_id: Optional[int] = None
    drift_auto: Optional[bool] = None
    laps: Optional[int] = None
    ghost_type: Optional[int] = None
    mii_name: Optional[str] = None
    lap_splits: Optional[List[int]] = None
    character_name: Optional[str] = None
    vehicle_name: Optional[str] = None
    input_data_size: Optional[int] = None
    total_input_frames: Optional[int] = None

CHARACTER_NAMES = {
    0: "Mario", 1: "Baby Peach", 2: "Waluigi", 3: "Bowser", 4: "Baby Daisy",
    5: "Dry Bones", 6: "Baby Mario", 7: "Luigi", 8: "Toad", 9: "Donkey Kong",
    10: "Yoshi", 11: "Wario", 12: "Baby Luigi", 13: "Toadette",
    14: "Koopa Troopa", 15: "Daisy", 16: "Peach", 17: "Birdo",
    18: "Diddy Kong", 19: "King Boo", 20: "Bowser Jr.", 21: "Dry Bowser",
    22: "Funky Kong", 23: "Rosalina", 24: "Mii Outfit A (Small)", 25: "Mii Outfit A (Small)",
    26: "Mii Outfit B (Small)", 27: "Mii Outfit B (Small)", 28: "Mii Outfit C (Small)", 29: "Mii Outfit C (Small)",
    30: "Mii Outfit A (Medium)", 31: "Mii Outfit A (Medium)", 32: "Mii Outfit B (Medium)", 33: "Mii Outfit B (Medium)",
    34: "Mii Outfit C (Medium)", 35: "Mii Outfit C (Medium)", 36: "Mii Outfit A (Large)", 37: "Mii Outfit A (Large)",
    38: "Mii Outfit B (Large)", 39: "Mii Outfit B (Large)", 40: "Mii Outfit C (Large)", 41: "Mii Outfit C (Large)",
    42: "Small Mii", 43: "Medium Mii", 44: "Large Mii", 45: "Peach",
    46: "Daisy", 47: "Rosalina"
}

VEHICLE_NAMES = {
    0: "Standard Kart S", 1: "Standard Kart M", 2: "Standard Kart L",
    3: "Booster Seat", 4: "Classic Dragster", 5: "Offroader",
    6: "Mini Beast", 7: "Wild Wing", 8: "Flame Flyer",
    9: "Cheep Charger", 10: "Super Blooper", 11: "Piranha Prowler",
    12: "Tiny Titan", 13: "Daystripper", 14: "Jetsetter",
    15: "Blue Falcon", 16: "Sprinter", 17: "Honeycoupe",
    18: "Standard Bike S", 19: "Standard Bike M", 20: "Standard Bike L",
    21: "Bullet Bike", 22: "Mach Bike", 23: "Flame Runner",
    24: "Bit Bike", 25: "Sugarscoot", 26: "Wario Bike",
    27: "Quacker", 28: "Zip Zip", 29: "Shooting Star",
    30: "Magikruiser", 31: "Sneakster", 32: "Spear",
    33: "Jet Bubble", 34: "Dolphin Dasher", 35: "Phantom"
}

def parse_mii_name(data: bytes, offset: int) -> str:
    try:
        if offset + 20 > len(data):
            return "Unknown Mii"
        name_bytes = data[offset:offset+20]
        utf8_bytes = bytes([name_bytes[i] for i in range(0, len(name_bytes), 2)])
        if len(utf8_bytes) > 0:
            try:
                name = utf8_bytes.decode('ascii', errors='replace')
                name = ''.join(c for c in name if c.isprintable())
                if (name and len(name) > 0 and len(name) <= 16 and 
                    name.replace(' ', '').isalnum()):
                    return name
            except Exception:
                pass
        return "Unknown Mii"
    except Exception:
        return "Unknown Mii"

def parse_lap_splits(data: bytes, offset: int, num_laps: int, finish_time_ms: int) -> List[int]:
    splits = []
    try:
        if num_laps == 1:
            splits.append(finish_time_ms)
        else:
            possible_offsets = [0x30, 0x34, 0x38, 0x3C, 0x40, 0x44, 0x48, 0x4C]
            for offset in possible_offsets:
                if offset + 4 <= len(data):
                    lap_time = int.from_bytes(data[offset:offset + 4], 'big')
                    if lap_time > 0 and lap_time < 600000:
                        splits.append(lap_time)
                        break
            if not splits and finish_time_ms > 0:
                avg_lap_time = finish_time_ms // num_laps
                splits = [avg_lap_time] * num_laps
    except Exception:
        pass
    return splits

def parse_input_data_info(data: bytes, compressed: bool, body_start: int) -> Tuple[int, int]:
    input_size = 0
    total_frames = 0
    try:
        if compressed:
            if body_start + 4 <= len(data):
                input_size = int.from_bytes(data[body_start:body_start + 4], 'big')
                total_frames = input_size // 2
        else:
            input_size = INPUT_MAX_UNCOMPRESSED
            total_frames = input_size // 2
    except Exception:
        pass
    return input_size, total_frames

def format_time_ms(ms: int) -> str:
    if ms is None or ms < 0:
        return "??:??.???"
    minutes = ms // 60000
    seconds = (ms % 60000) // 1000
    millis = ms % 1000
    return f"{minutes}:{seconds:02d}.{millis:03d}"

def get_character_name(char_id: int) -> str:
    return CHARACTER_NAMES.get(char_id, f"Unknown ({char_id})")

def get_vehicle_name(vehicle_id: int) -> str:
    return VEHICLE_NAMES.get(vehicle_id, f"Unknown ({vehicle_id})")

def _get_bits(buf: bytes, bit_offset: int, bit_len: int) -> int:
    byte_start = bit_offset // 8
    byte_end = (bit_offset + bit_len + 7) // 8
    subset = int.from_bytes(buf[byte_start:byte_end], "big")
    shift = (8 * (byte_end - byte_start)) - ((bit_offset + bit_len) - 8 * byte_start)
    subset >>= shift
    mask = (1 << bit_len) - 1
    return subset & mask

def parse_finish_time_ms(header: bytes) -> int:
    minutes = _get_bits(header, 0x04 * 8 + 0, 7)
    seconds = _get_bits(header, 0x04 * 8 + 7, 7)
    millis = _get_bits(header, 0x05 * 8 + 6, 10)
    return (minutes * 60 + seconds) * 1000 + millis

def validate_rkg(data: bytes) -> RkgInfo:
    if len(data) < RKG_HEADER_SIZE + 4:
        return RkgInfo(False, "File too small to be a valid RKG (missing header/CRC)")
    if data[:4] != RKG_MAGIC:
        return RkgInfo(False, "Invalid magic; expected 'RKGD'")
    header = data[:RKG_HEADER_SIZE]
    finish_ms = parse_finish_time_ms(header)
    track_id = _get_bits(header, 0x07 * 8 + 0, 6)
    vehicle_id = _get_bits(header, 0x08 * 8 + 0, 6)
    character_id = _get_bits(header, 0x08 * 8 + 6, 6)
    year_rel = _get_bits(header, 0x09 * 8 + 4, 7)
    month = _get_bits(header, 0x0A * 8 + 3, 4)
    day = _get_bits(header, 0x0A * 8 + 7, 5)
    controller_id = _get_bits(header, 0x0B * 8 + 4, 4)
    compressed_flag = bool(_get_bits(header, 0x0C * 8 + 4, 1))
    ghost_type = _get_bits(header, 0x0C * 8 + 7, 7)
    drift_auto = bool(_get_bits(header, 0x0D * 8 + 6, 1))
    input_len_uncompressed = int.from_bytes(header[0x0E:0x10], "big")
    laps = header[0x10]
    if not (1 <= month <= 12):
        return RkgInfo(False, f"Invalid month {month}")
    if not (1 <= day <= 31):
        return RkgInfo(False, f"Invalid day {day}")
    if input_len_uncompressed > INPUT_MAX_UNCOMPRESSED:
        return RkgInfo(False, f"Uncompressed input length too large: {input_len_uncompressed}")
    body = data[RKG_HEADER_SIZE:-4]
    crc_stored_be = int.from_bytes(data[-4:], "big")
    crc_stored_le = int.from_bytes(data[-4:], "little")
    crc_calc = binascii.crc32(data[:-4]) & 0xFFFFFFFF
    if compressed_flag:
        if len(body) < 8:
            return RkgInfo(False, "Compressed ghost but input section too small")
        comp_len = int.from_bytes(body[:4], "big")
        if comp_len > INPUT_MAX_UNCOMPRESSED:
            return RkgInfo(False, f"Compressed data section too large: {comp_len}")
        if len(body) < 4 + comp_len:
            return RkgInfo(False, "Truncated compressed data")
    else:
        if len(body) < INPUT_MAX_UNCOMPRESSED:
            return RkgInfo(False, "Uncompressed ghost missing padded input section")
    if crc_calc != crc_stored_be and crc_calc != crc_stored_le:
        return RkgInfo(False, "CRC32 mismatch")
    mii_name = parse_mii_name(header, 0x3F)
    lap_splits = parse_lap_splits(header, 0x30, laps, finish_ms)
    input_size, total_frames = parse_input_data_info(data, compressed_flag, RKG_HEADER_SIZE)
    year = 2000 + year_rel
    return RkgInfo(
        True,
        "OK",
        compressed=compressed_flag,
        finish_time_ms=finish_ms,
        track_id=track_id,
        vehicle_id=vehicle_id,
        character_id=character_id,
        year=year,
        month=month,
        day=day,
        controller_id=controller_id,
        drift_auto=drift_auto,
        laps=laps,
        ghost_type=ghost_type,
        mii_name=mii_name,
        lap_splits=lap_splits,
        character_name=get_character_name(character_id),
        vehicle_name=get_vehicle_name(vehicle_id),
        input_data_size=input_size,
        total_input_frames=total_frames,
    )

def parse_folder_map(text: str) -> Dict[str, str]:
    mapping: Dict[str, str] = {}
    for line in text.splitlines():
        line = line.strip()
        if not line or line.startswith("#"):
            continue
        m = re.match(r"^(.*?)\s*=\s*([0-9A-Fa-f]{8})\s*$", line)
        if m:
            name = m.group(1).strip()
            h = m.group(2).upper()
            mapping[name] = h
    return mapping

class AutoComplete(ttk.Frame):
    def __init__(self, master, values: List[str]):
        super().__init__(master)
        self.var = tk.StringVar()
        self.entry = ttk.Entry(self, textvariable=self.var)
        self.entry.pack(fill=tk.X)
        self.listbox = tk.Listbox(self, height=6)
        self.listbox.pack(fill=tk.BOTH, expand=True)
        self.all_values = sorted(values)
        self.filtered = self.all_values
        self.var.trace_add("write", self._on_change)
        self.listbox.bind("<<ListboxSelect>>", self._on_select)
        self.listbox.bind("<Double-Button-1>", self._on_double)
        self.update_list("")

    def _on_change(self, *_):
        self.update_list(self.var.get())

    def _on_select(self, *_):
        sel = self.get_selected()
        if sel:
            self.var.set(sel)

    def _on_double(self, *_):
        sel = self.get_selected()
        if sel:
            self.var.set(sel)
            self.event_generate("<<AutoCompleteChosen>>")

    def update_values(self, values: List[str]):
        self.all_values = sorted(values)
        self.update_list(self.var.get())

    def update_list(self, prefix: str):
        prefix_low = prefix.lower()
        self.filtered = [v for v in self.all_values if prefix_low in v.lower()]
        self.listbox.delete(0, tk.END)
        for v in self.filtered[:200]:
            self.listbox.insert(tk.END, v)

    def get(self) -> str:
        return self.var.get().strip()

    def get_selected(self) -> Optional[str]:
        try:
            idx = self.listbox.curselection()
            if idx:
                return self.filtered[idx[0]]
        except Exception:
            pass
        return None

class App(tk.Tk):
    def __init__(self):
        super().__init__()
        self.title("IKW Ghost Manager")
        self.geometry("900x600")
        self.settings_pul_path: Optional[str] = None
        self.ghosts_root: Optional[str] = None
        self.track_map: Dict[str, str] = {}
        self._build_ui()
        self.fetch_track_list()

    def _build_ui(self):
        top = ttk.Frame(self)
        top.pack(fill=tk.X, padx=10, pady=10)
        ttk.Label(top, text="Settings.pul").grid(row=0, column=0, sticky=tk.W, pady=(8,0))
        ttk.Button(top, text="Select Settings.pul", command=self.pick_settings_pul).grid(row=0, column=1, sticky=tk.W, pady=(8,0))
        self.ghosts_label = ttk.Label(top, text="Ghosts folder: not set")
        self.ghosts_label.grid(row=0, column=2, sticky=tk.W, pady=(8,0))
        sep = ttk.Separator(self)
        sep.pack(fill=tk.X, pady=6)
        mid = ttk.Frame(self)
        mid.pack(fill=tk.BOTH, expand=True, padx=10)
        left = ttk.Labelframe(mid, text="Find & Download Ghosts")
        left.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(0,6))
        right = ttk.Labelframe(mid, text="Validate & Export Ghosts")
        right.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        ttk.Label(left, text="Track").pack(anchor=tk.W, padx=8, pady=(6,0))
        self.ac = AutoComplete(left, [])
        self.ac.pack(fill=tk.BOTH, expand=True, padx=8)
        ttk.Label(left, text="Mode").pack(anchor=tk.W, padx=8, pady=(6,0))
        self.mode_var = tk.StringVar(value=VALID_MODES[0])
        self.mode_combo = ttk.Combobox(left, values=VALID_MODES, textvariable=self.mode_var, state="readonly")
        self.mode_combo.pack(fill=tk.X, padx=8)
        ttk.Button(left, text="List Ghosts on Server", command=self.list_remote_ghosts).pack(padx=8, pady=6, anchor=tk.W)
        self.remote_list = tk.Listbox(left)
        self.remote_list.pack(fill=tk.BOTH, expand=True, padx=8, pady=(0,8))
        self.remote_list.bind("<Double-Button-1>", lambda e: self.preview_selected())
        button_frame = ttk.Frame(left)
        button_frame.pack(fill=tk.X, padx=8, pady=(0,8))
        ttk.Button(button_frame, text="Preview Selected", command=self.preview_selected).pack(side=tk.LEFT, padx=(0,6))
        ttk.Button(button_frame, text="Download Selected", command=self.download_selected).pack(side=tk.LEFT)
        ttk.Label(right, text="Local RKG file").pack(anchor=tk.W, padx=8, pady=(6,0))
        ttk.Button(right, text="Find Local Ghost", command=self.pick_local_ghost).pack(anchor=tk.W, padx=8)
        self.rkg_info = tk.Text(right, height=12)
        self.rkg_info.pack(fill=tk.BOTH, expand=True, padx=8, pady=6)
        up_bottom = ttk.Frame(right)
        up_bottom.pack(fill=tk.X, padx=8, pady=6)
        ttk.Label(up_bottom, text="Track").grid(row=0, column=0, sticky=tk.W)
        self.up_track_ac = AutoComplete(up_bottom, [])
        self.up_track_ac.entry.config(width=40)
        self.up_track_ac.grid(row=0, column=1, sticky=tk.W, padx=(6,16))
        ttk.Label(up_bottom, text="Mode").grid(row=0, column=2, sticky=tk.W)
        self.up_mode = ttk.Combobox(up_bottom, values=VALID_MODES, state="readonly")
        self.up_mode.grid(row=0, column=3, sticky=tk.W, padx=(6,16))
        ttk.Button(up_bottom, text="Export Ghost", command=self.export_current).grid(row=0, column=4, sticky=tk.W)
        self.status = tk.StringVar(value="Ready")
        ttk.Label(self, textvariable=self.status, relief=tk.SUNKEN, anchor=tk.W).pack(fill=tk.X)

    def pick_settings_pul(self):
        path = filedialog.askopenfilename(title="Select Settings.pul", filetypes=[("Settings.pul", "Settings.pul"), ("All files", "*.*")])
        if not path:
            return
        if os.path.basename(path) != "Settings.pul":
            messagebox.showerror("Invalid file", "Please select the Settings.pul file.")
            return
        self.settings_pul_path = path
        base = os.path.dirname(path)
        ghosts = os.path.join(base, "ghosts")
        self.ghosts_root = ghosts
        os.makedirs(ghosts, exist_ok=True)
        self.ghosts_label.config(text=f"Ghosts folder: {ghosts}")
        self.status.set("Ghosts folder set.")

    def fetch_track_list(self):
        url = parse.urljoin(DEFAULT_SERVER, FOLDER_MAP_TXT)
        self.status.set("Downloading track list…")
        self.update_idletasks()
        try:
            data = http_get(url)
            mapping = parse_folder_map(data.decode("utf-8", "replace"))
            if not mapping:
                raise ValueError("No tracks parsed from FolderToTrackName.txt")
            self.track_map = mapping
            names = sorted(mapping.keys())
            self.ac.update_values(names)
            self.up_track_ac.update_values(names)
            self.status.set(f"Loaded {len(names)} tracks from server.")
        except Exception as e:
            messagebox.showerror("Error", f"Failed to fetch track list:\n{e}")
            self.status.set("Failed to fetch track list.")

    def _require_ready(self) -> bool:
        if not self.track_map:
            messagebox.showwarning("Missing data", "Please fetch the track list first.")
            return False
        if not self.ghosts_root:
            messagebox.showwarning("Missing folder", "Please select Settings.pul to set your ghosts folder.")
            return False
        return True

    def list_remote_ghosts(self):
        if not self._require_ready():
            return
        track = self.ac.get()
        if track not in self.track_map:
            messagebox.showerror("Track not found", "Please choose a valid track from the list.")
            return
        mode = self.mode_var.get()
        if mode not in VALID_MODES:
            messagebox.showerror("Invalid mode", "Choose a valid mode.")
            return
        h = self.track_map[track]
        base = DEFAULT_SERVER.rstrip("/") + "/"
        url = parse.urljoin(base, f"{h}/{mode}/")
        self.status.set("Listing ghosts…")
        self.update_idletasks()
        files = http_list(url)
        rkgs = [f for f in files if f.lower().endswith(".rkg")]
        self.remote_list.delete(0, tk.END)
        for f in rkgs:
            self.remote_list.insert(tk.END, f)
        self.status.set(f"Found {len(rkgs)} ghosts for {track} [{mode}].")

    def download_selected(self):
        if not self._require_ready():
            return
        sel = self.remote_list.curselection()
        if not sel:
            messagebox.showwarning("Nothing selected", "Select a ghost to download.")
            return
        filename = self.remote_list.get(sel[0])
        track = self.ac.get()
        h = self.track_map.get(track)
        mode = self.mode_var.get()
        base = DEFAULT_SERVER.rstrip("/") + "/"
        url = parse.urljoin(base, f"{h}/{mode}/{filename}")
        self.status.set("Downloading ghost for preview…")
        self.update_idletasks()
        try:
            blob = http_get(url)
            info = validate_rkg(blob)
            preview_text = self._format_ghost_preview(filename, track, mode, info)
            if not messagebox.askyesno("Ghost Preview", 
                f"Preview of ghost: {filename}\n\n{preview_text}\n\nDownload this ghost?"):
                self.status.set("Download cancelled.")
                return
            self.status.set("Saving ghost…")
            self.update_idletasks()
            if not info.valid:
                if not messagebox.askyesno("Validation warning", f"Downloaded file failed validation: {info.reason}\nSave anyway?"):
                    self.status.set("Download aborted.")
                    return
            local_dir = os.path.join(self.ghosts_root, h, mode)
            os.makedirs(local_dir, exist_ok=True)
            safe_name = filename.replace(" ", "_")
            if not SAFE_FILENAME_RE.match(safe_name):
                safe_name = re.sub(r"[^A-Za-z0-9._-]", "_", safe_name)
            local_path = os.path.join(local_dir, safe_name)
            with open(local_path, "wb") as f:
                f.write(blob)
            self.status.set(f"Saved to {local_path}")
            messagebox.showinfo("Saved", f"Ghost saved to:\n{local_path}")
        except Exception as e:
            messagebox.showerror("Error", f"Download failed:\n{e}")
            self.status.set("Download failed.")

    def preview_selected(self):
        if not self._require_ready():
            return
        sel = self.remote_list.curselection()
        if not sel:
            messagebox.showwarning("Nothing selected", "Select a ghost to preview.")
            return
        filename = self.remote_list.get(sel[0])
        track = self.ac.get()
        h = self.track_map.get(track)
        mode = self.mode_var.get()
        base = DEFAULT_SERVER.rstrip("/") + "/"
        url = parse.urljoin(base, f"{h}/{mode}/{filename}")
        self.status.set("Downloading ghost for preview…")
        self.update_idletasks()
        try:
            blob = http_get(url)
            info = validate_rkg(blob)
            preview_text = self._format_ghost_preview(filename, track, mode, info)
            messagebox.showinfo("Ghost Preview", f"Preview of ghost: {filename}\n\n{preview_text}")
            self.status.set("Preview completed.")
        except Exception as e:
            messagebox.showerror("Error", f"Preview failed:\n{e}")
            self.status.set("Preview failed.")

    def _format_ghost_preview(self, filename: str, track: str, mode: str, info: RkgInfo) -> str:
        lines = [
            f"Track: {track}",
            f"Mode: {mode}",
            f"Filename: {filename}",
            "",
            "=== Ghost Details ===",
            f"Finish Time: {format_time_ms(info.finish_time_ms)}",
            f"Character: {info.character_name}",
            f"Vehicle: {info.vehicle_name}",
            f"Date: {info.year}-{info.month:02d}-{info.day:02d}",
            f"Laps: {info.laps}",
            f"Controller: {info.controller_id}",
            f"Hybrid Drift: {info.drift_auto}",
        ]
        if info.mii_name and info.mii_name != "Unknown Mii":
            lines.append(f"Mii Creator: {info.mii_name}")
        if info.lap_splits:
            lines.append("")
            lines.append("=== Lap Splits ===")
            for i, split in enumerate(info.lap_splits):
                lines.append(f"Lap {i+1}: {format_time_ms(split)}")
        if not info.valid:
            lines.append("")
            lines.append(f"⚠️ VALIDATION WARNING: {info.reason}")
        return "\n".join(lines)

    def pick_local_ghost(self):
        if not self._require_ready():
            return
        track = self.up_track_ac.get()
        mode = self.up_mode.get()
        if track not in self.track_map or mode not in VALID_MODES:
            messagebox.showerror("Missing info", "Select a valid track and mode first.")
            return
        h = self.track_map[track]
        local_dir = os.path.join(self.ghosts_root, h, mode)
        if not os.path.isdir(local_dir):
            messagebox.showwarning("No ghosts", f"No folder found for {track} [{mode}].")
            return
        files = [f for f in os.listdir(local_dir) if f.lower().endswith(".rkg")]
        if not files:
            messagebox.showwarning("No ghosts", f"No RKG files found for {track} [{mode}].")
            return
        if len(files) > 1:
            sel_win = tk.Toplevel(self)
            sel_win.title("Select Ghost File")
            tk.Label(sel_win, text="Select a ghost file:").pack(padx=8, pady=8)
            lb = tk.Listbox(sel_win, height=10)
            for f in files:
                lb.insert(tk.END, f)
            lb.pack(fill=tk.BOTH, expand=True, padx=8, pady=8)
            chosen = {"file": None}
            def choose_and_close():
                sel = lb.curselection()
                if sel:
                    chosen["file"] = files[sel[0]]
                    sel_win.destroy()
            ttk.Button(sel_win, text="Select", command=choose_and_close).pack(pady=8)
            lb.bind("<Double-Button-1>", lambda e: choose_and_close())
            sel_win.grab_set()
            self.wait_window(sel_win)
            if not chosen["file"]:
                return
            filename = chosen["file"]
        else:
            filename = files[0]
        path = os.path.join(local_dir, filename)
        try:
            with open(path, "rb") as f:
                data = f.read()
            info = validate_rkg(data)
            self._show_rkg_info(path, data, info)
            self._current_export = (path, data, info)
        except Exception as e:
            messagebox.showerror("Error", f"Failed to read/validate RKG:\n{e}")

    def _show_rkg_info(self, path: str, data: bytes, info: RkgInfo):
        self.rkg_info.delete("1.0", tk.END)
        size = len(data)
        lines = [
            f"File: {path}",
            f"Size: {size} bytes",
            f"Validation: {'OK' if info.valid else 'FAILED'} ({info.reason})",
            "",
            "=== Basic Information ===",
            f"Compressed: {info.compressed}",
            f"Finish Time: {format_time_ms(info.finish_time_ms)}",
            f"Track ID: {info.track_id}",
            f"Vehicle: {info.vehicle_name} (ID: {info.vehicle_id})",
            f"Character: {info.character_name} (ID: {info.character_id})",
            f"Date: {info.year}-{info.month:02d}-{info.day:02d}",
            f"Controller: {info.controller_id}  Hybrid Drift: {info.drift_auto}",
            f"Laps: {info.laps}  Ghost Type: {info.ghost_type}",
            "",
            "=== Enhanced Preview ===",
            f"Mii Name: {info.mii_name}",
            f"Input Data Size: {info.input_data_size} bytes",
            f"Total Input Frames: {info.total_input_frames}",
        ]
        if info.lap_splits:
            lines.append("")
            lines.append("=== Lap Splits ===")
            for i, split in enumerate(info.lap_splits):
                lines.append(f"Lap {i+1}: {format_time_ms(split)}")
        name_ok = SAFE_FILENAME_RE.match(os.path.basename(path)) is not None and " " not in os.path.basename(path)
        if not name_ok:
            lines.append("")
            lines.append("WARNING: Filename contains spaces or disallowed chars; game may ignore it.")
        self.rkg_info.insert("1.0", "\n".join(lines))

    def export_current(self):
        cur = getattr(self, "_current_export", None)
        if not cur:
            messagebox.showwarning("No file", "Choose a local RKG to export first.")
            return
        path, data, info = cur
        if not info.valid:
            if not messagebox.askyesno("Validation warning", f"File failed validation: {info.reason}\nExport anyway?"):
                self.status.set("Export aborted.")
                return
        filename = os.path.basename(path)
        safe_name = filename.replace(" ", "_")
        if not SAFE_FILENAME_RE.match(safe_name):
            safe_name = re.sub(r"[^A-Za-z0-9._-]", "_", safe_name)
        save_path = filedialog.asksaveasfilename(initialfile=safe_name, defaultextension=".rkg", filetypes=[("RKG files", "*.rkg")])
        if not save_path:
            self.status.set("Export cancelled.")
            return
        self.status.set("Exporting…")
        self.update_idletasks()
        try:
            with open(save_path, "wb") as f:
                f.write(data)
            messagebox.showinfo("Exported", f"Ghost exported to:\n{save_path}")
            self.status.set("Export successful.")
        except Exception as e:
            messagebox.showerror("Export failed", f"Failed to export:\n{e}")
            self.status.set("Export failed.")

if __name__ == "__main__":
    app = App()
    app.mainloop()